<?php

namespace App\Actions\Saas;

use App\Models\Employee\Employee;
use App\Models\Saas\Plan;
use App\Models\Team;
use Illuminate\Validation\ValidationException;

class GetLimit
{
    public function execute(Plan $plan, bool $validate = false): array
    {
        $teamCount = Team::count();
        $employeeCount = Employee::count();

        $maxTeamLimit = (int) $plan->getFeature('max_team_limit');
        $teamWiseLimit = (bool) $plan->getFeature('team_wise_limit');
        $maxEmployeePerTeamLimit = (int) $plan->getFeature('max_employee_per_team_limit');
        $maxEmployeeLimit = $teamWiseLimit ? ($teamCount * $maxEmployeePerTeamLimit) : (int) $plan->getFeature('max_employee_limit');

        $teamUsagePercentage = round($teamCount / $maxTeamLimit * 100, 2);
        $employeeUsagePercentage = round($employeeCount / $maxEmployeeLimit * 100, 2);

        if ($validate) {
            if ($teamUsagePercentage > 100) {
                throw ValidationException::withMessages(['message' => trans('saas.subscription.team_limit_crossed')]);
            }

            if ($employeeUsagePercentage > 100) {
                throw ValidationException::withMessages(['message' => trans('saas.subscription.employee_limit_crossed')]);
            }
        }

        if ($teamUsagePercentage > 100) {
            $teamUsagePercentage = 100;
        }

        if ($employeeUsagePercentage > 100) {
            $employeeUsagePercentage = 100;
        }

        return [
            [
                'label' => trans('team.team'),
                'percent' => $teamUsagePercentage,
                'value' => $teamCount,
                'max' => $maxTeamLimit,
                'color' => $this->getPercentageColor($teamUsagePercentage),
            ],
            [
                'label' => trans('employee.employee'),
                'percent' => $employeeUsagePercentage,
                'value' => $employeeCount,
                'max' => $maxEmployeeLimit,
                'color' => $this->getPercentageColor($employeeUsagePercentage),
            ],
        ];
    }

    private function getPercentageColor($percent)
    {
        return match (true) {
            $percent <= 10 => 'bg-info',
            $percent > 10 && $percent <= 50 => 'bg-success',
            $percent > 50 && $percent <= 80 => 'bg-warning',
            $percent > 80 => 'bg-danger',
        };
    }
}
