<?php

namespace App\Actions\Saas\Subscription;

use App\Actions\Saas\GetLimit;
use App\Models\Saas\Plan;
use App\Models\Saas\Subscription;
use App\Models\Saas\Tenant;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class Calculate
{
    public function execute(Request $request)
    {
        $planUuid = $request->plan;
        $frequency = $request->frequency;
        $currency = $request->currency;

        $tenant = Tenant::query()
            ->withActiveSubscriptionId()
            ->find(config('saas.tenant.id'));

        $plan = Plan::find($tenant->plan_id);

        $newPlan = Plan::query()
            ->whereUuid($planUuid)
            ->first();

        $existingSubscriptions = Subscription::query()
            ->whereTenantId($tenant->id)
            ->where('amount', '>', 0)
            ->where('status', '=', 1)
            ->get();

        (new GetLimit)->execute(plan: $newPlan, validate: true);

        $request->merge(['plan' => $newPlan]);

        // Free Subscription
        if ($newPlan->getFeature('is_free')) {
            if ($plan->uuid == $newPlan->uuid) {
                throw ValidationException::withMessages(['message' => trans('saas.subscription.same_free_plan')]);
            }

            return (new CalculateFreeSubscription)->execute(tenant: $tenant);
        }

        // New Subscription
        if (! $existingSubscriptions->count()) {
            return (new CalculateNewSubscription)->execute(
                request: $request,
                tenant: $tenant,
                newPlan: $newPlan,
                existingSubscriptions: $existingSubscriptions
            );
        }

        // Renew Subscription

        elseif ($existingSubscriptions->count() && $tenant->expiry_date->value < today()->toDateString()) {
            return (new CalculateRenewSubscription)->execute(
                request: $request,
                tenant: $tenant,
                newPlan: $newPlan,
                existingSubscriptions: $existingSubscriptions
            );
        }

        // Continue Subscription

        elseif ($plan->uuid == $planUuid) {
            return (new CalculateContinueSubscription)->execute(
                request: $request,
                tenant: $tenant,
                newPlan: $newPlan,
                existingSubscriptions: $existingSubscriptions
            );
        }

        // Sibling Subscription

        elseif ($plan->getPrice($currency, $frequency) == $newPlan->getPrice($currency, $frequency)) {
            return (new CalculateSiblingSubscription)->execute(
                request: $request,
                tenant: $tenant,
                newPlan: $newPlan,
                existingSubscriptions: $existingSubscriptions
            );
        }

        // Upgrade Subscription

        elseif ($plan->getPrice($currency, $frequency) < $newPlan->getPrice($currency, $frequency)) {
            return (new CalculateUpgradeSubscription)->execute(
                request: $request,
                tenant: $tenant,
                newPlan: $newPlan,
                existingSubscriptions: $existingSubscriptions
            );
        }

        // Downgrade Subscription

        elseif ($plan->getPrice($currency, $frequency) > $newPlan->getPrice($currency, $frequency)) {
            return (new CalculateDowngradeSubscription)->execute(
                request: $request,
                tenant: $tenant,
                newPlan: $newPlan,
                existingSubscriptions: $existingSubscriptions
            );
        } else {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_operation')]);
        }
    }
}
