<?php

namespace App\Console\Commands;

use App\Actions\Saas\SwitchDatabase;
use App\Concerns\GetTenantsForCommand;
use App\Models\Media;
use Illuminate\Console\Command;

class MediaPrune extends Command
{
    use GetTenantsForCommand;

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'media:prune {tenant?* : ID of Tenants} {--force}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Prune unused media';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $force = $this->option('force');

        if (\App::environment('production') && ! $force) {
            $this->error('Could not sync in production mode');
            exit;
        }

        $tenantIds = $this->argument('tenant');

        $tenants = $this->getTenants($tenantIds);

        $bar = $this->output->createProgressBar(count($tenants));
        $bar->start();

        activity()->disableLogging();

        foreach ($tenants as $tenant) {
            $this->newLine();
            $this->info('Pruning tenant: '.$tenant->id);

            (new SwitchDatabase)->execute($tenant->database);

            $media = Media::query()
                ->select('name', 'id')
                ->where('created_at', '<', now()->subHour(1)->toDateTimeString())
                ->where(function ($q) {
                    $q->whereNull('model_id')
                        ->orWhere('status', 0);
                })
                ->get();

            $media->each(function (Media $media) {
                if (\Storage::exists($media->name)) {
                    \Storage::delete($media->name);
                }
            });

            Media::whereIn('id', $media->pluck('id')->all())->delete();

            $this->info($media->count().' unused media pruned.');
            $this->info('Pruning complete tenant: '.$tenant->id);

            $bar->advance();
            $this->newLine();

            (new SwitchDatabase)->execute();
        }

        $bar->finish();

        activity()->enableLogging();

        $this->info('Pruning complete.');
    }
}
