<?php

namespace App\Http\Requests\Leave;

use App\Concerns\CustomFormFieldValidation;
use App\Enums\CustomFieldForm;
use App\Helpers\CalHelper;
use App\Models\CustomField;
use App\Models\Employee\Employee;
use App\Models\Leave\Allocation as LeaveAllocation;
use App\Models\Leave\Request as LeaveRequest;
use App\Models\Leave\Type as LeaveType;
use App\Models\Payroll\Payroll;
use Carbon\Carbon;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;

class RequestRequest extends FormRequest
{
    use CustomFormFieldValidation;

    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'leave_type' => 'required|uuid',
            'start_date' => 'required|date_format:Y-m-d',
            'end_date' => 'required|date|after_or_equal:start_date',
            'is_half_day' => 'boolean',
            'reason' => 'required|min:10|max:1000',
        ];
    }

    public function withValidator($validator)
    {
        if (! $validator->passes()) {
            return;
        }

        $validator->after(function ($validator) {
            $uuid = $this->route('leave_request');

            $attendancePastDayLimit = config('config.leave.attendance_past_day_limit');

            $allowedPastDate = Carbon::now()->subDays($attendancePastDayLimit)->toDateString();

            if ($this->start_date < $allowedPastDate) {
                throw ValidationException::withMessages(['start_date' => trans('leave.request.past_date_not_allowed', ['attribute' => $attendancePastDayLimit])]);
            }

            $employee = Employee::auth()->first();

            $leaveType = LeaveType::query()
                ->byTeam()
                ->whereUuid($this->leave_type)
                ->getOrFail(trans('leave.type.type'), 'leave_type');

            $dateDiff = CalHelper::dateDiff($this->start_date, $this->end_date);

            if ($this->is_half_day && $dateDiff > 1) {
                throw ValidationException::withMessages(['message' => trans('leave.request.half_day_invalid')]);
            }

            if ($this->is_half_day && ! config('config.leave.allow_employee_half_day_leave')) {
                throw ValidationException::withMessages(['message' => trans('leave.request.half_day_not_allowed')]);
            }

            $overlappingRequest = LeaveRequest::query()
                ->whereEmployeeId($employee->id)
                ->when($uuid, function ($q, $uuid) {
                    $q->where('uuid', '!=', $uuid);
                })
                ->betweenPeriod($this->start_date, $this->end_date)
                ->count();

            if ($overlappingRequest) {
                $validator->errors()->add('message', trans('leave.request.range_exists', ['start' => CalHelper::showDate($this->start_date), 'end' => CalHelper::showDate($this->end_date)]));
            }

            $duration = CalHelper::dateDiff($this->start_date, $this->end_date);

            $payrollGenerated = Payroll::query()
                ->whereEmployeeId($employee->id)
                ->betweenPeriod($this->start_date, $this->end_date)
                ->exists();

            if ($payrollGenerated) {
                throw ValidationException::withMessages(['message' => trans('leave.request.could_not_perform_if_payroll_generated')]);
            }

            $requestWithExhaustedCredit = config('config.leave.allow_employee_request_leave_with_exhausted_credit');

            $query = LeaveAllocation::query()
                ->with('records')
                ->whereEmployeeId($employee->id)
                ->where('start_date', '<=', $this->start_date)
                ->where('end_date', '>=', $this->end_date);

            if ($requestWithExhaustedCredit) {
                $leaveAllocation = $query->first();
            } else {
                $leaveAllocation = $query->getOrFail(trans('leave.allocation.allocation'));
            }

            if (! $requestWithExhaustedCredit) {
                $leaveAllocationRecord = $leaveAllocation->records->where('leave_type_id', $leaveType->id)->hasOrFail(trans('leave.type.no_allocation_found'));

                $balance = $leaveAllocationRecord->allotted - $leaveAllocationRecord->used;

                if ($balance < $duration) {
                    throw ValidationException::withMessages(['message' => trans('leave.type.balance_exhausted', ['balance' => $balance, 'duration' => $duration])]);
                }
            }

            $this->merge([
                'employee_id' => $employee->id,
                'leave_type_id' => $leaveType->id,
                'duration' => $duration,
                'leave_allocation_id' => $leaveAllocation?->id,
            ]);

            $customFields = CustomField::query()
                ->byTeam()
                ->whereForm(CustomFieldForm::LEAVE_REQUEST)
                ->get();

            $newCustomFields = $this->validateFields($validator, $customFields, $this->input('custom_fields', []));

            $this->merge([
                'custom_fields' => $newCustomFields,
            ]);
        });
    }

    /**
     * Translate fields with user friendly name.
     *
     * @return array
     */
    public function attributes()
    {
        return [
            'leave_type' => __('leave.type.type'),
            'start_date' => __('leave.request.props.start_date'),
            'end_date' => __('leave.request.props.end_date'),
            'reason' => __('leave.request.props.reason'),
        ];
    }

    /**
     * Get the error messages for the defined validation rules.
     *
     * @return array
     */
    public function messages()
    {
        return [];
    }
}
