<?php

namespace App\Models\Communication;

use App\Concerns\HasConfig;
use App\Concerns\HasFilter;
use App\Concerns\HasMedia;
use App\Concerns\HasMeta;
use App\Concerns\HasUuid;
use App\Enums\Communication\Type;
use App\Models\Audience;
use App\Models\Team;
use App\Models\User;
use App\Scopes\AudienceScope;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;

class Communication extends Model
{
    use AudienceScope, HasConfig, HasFactory, HasFilter, HasMedia, HasMeta, HasUuid, LogsActivity;

    protected $guarded = [];

    protected $primaryKey = 'id';

    protected $table = 'communications';

    protected $attributes = [];

    protected $casts = [
        'type' => Type::class,
        'recipients' => 'array',
        'lists' => 'array',
        'audience' => 'array',
        'config' => 'array',
        'meta' => 'array',
    ];

    public function getModelName(): string
    {
        return 'Communication';
    }

    public function team(): BelongsTo
    {
        return $this->belongsTo(Team::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function audiences()
    {
        return $this->morphMany(Audience::class, 'shareable');
    }

    public function scopeFilterAccessible(Builder $query)
    {
        $query->where(function ($q) {
            $q->accessible();
        });
    }

    public function scopeFindByUuidOrFail(Builder $query, string $uuid)
    {
        return $query
            ->byTeam()
            ->filterAccessible()
            ->whereUuid($uuid)
            ->getOrFail(trans('communication.communication'));
    }

    public function scopeFindEmailByUuidOrFail(Builder $query, string $uuid)
    {
        return $query
            ->byTeam()
            ->filterAccessible()
            ->whereType(Type::EMAIL)
            ->whereUuid($uuid)
            ->getOrFail(trans('communication.email.email'));
    }

    public function scopeByTeam(Builder $query, ?int $teamId = null)
    {
        $teamId = $teamId ?? auth()->user()?->current_team_id;

        $query->where('team_id', $teamId);
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->useLogName('communication')
            ->logAll()
            ->logExcept(['updated_at'])
            ->logOnlyDirty();
    }
}
