<?php

namespace App\Models\Saas;

use App\Casts\DateCast;
use App\Concerns\HasFilter;
use App\Concerns\HasMeta;
use App\Enums\Saas\PlanFrequency;
use App\Helpers\CalHelper;
use App\ValueObjects\Percent;
use App\ValueObjects\Price;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Arr;
use Illuminate\Validation\ValidationException;

class Subscription extends Model
{
    use HasFactory, HasFilter, HasMeta;

    protected $table = 'subscriptions';

    protected $casts = [
        'frequency' => PlanFrequency::class,
        'start_date' => DateCast::class,
        'expiry_date' => DateCast::class,
        'meta' => 'array',
        'tax_details' => 'array',
        'billing_address' => 'array',
        'payment_gateway' => 'array',
        'failed_logs' => 'array',
    ];

    protected $connection = 'landlord';

    protected function amount(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => \Price::from($value, $this->currency),
        );
    }

    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    public function plan(): BelongsTo
    {
        return $this->belongsTo(Plan::class);
    }

    public function isAccessible(): void
    {
        if ($this->tenant_id != config('saas.tenant.id')) {
            throw ValidationException::withMessages(['message' => trans('user.errors.permission_denied')]);
        }
    }

    public function getCustomerAddressAttribute(): string
    {
        return Arr::implode(Arr::notEmpty([
            Arr::get($this->billing_address, 'address.address_line1'),
            Arr::get($this->billing_address, 'address.address_line2'),
            Arr::get($this->billing_address, 'address.city'),
            Arr::get($this->billing_address, 'address.state'),
            Arr::get($this->billing_address, 'address.zipcode'),
            Arr::get($this->billing_address, 'address.country'),
        ]));
    }

    public function getPeriodAttribute(): string
    {
        return CalHelper::getPeriod($this->start_date->value, $this->expiry_date->value);
    }

    public function getTaxAmountAttribute(): Price
    {
        return \Price::from(Arr::get($this->tax_details, 'amount', 0), $this->currency);
    }

    public function getAmountWithoutTaxAttribute(): Price
    {
        return \Price::from($this->amount->value - $this->tax_amount->value, $this->currency);
    }

    public function getTaxLabelAttribute(): ?string
    {
        return Arr::get($this->tax_details, 'label');
    }

    public function getTaxRateAttribute(): Percent
    {
        return \Percent::from(Arr::get($this->tax_details, 'rate'));
    }

    public function getTaxTypeAttribute(): string
    {
        return Arr::get($this->tax_details, 'is_exclusive') ? trans('saas.subscription.props.tax_type_exclusive') : trans('saas.subscription.props.tax_type_inclusive');
    }
}
