<?php

namespace App\Services\Employee;

use App\Actions\CreateContact;
use App\Actions\SendMailTemplate;
use App\Actions\UpdateContact;
use App\Concerns\Auth\EnsureUniqueUserEmail;
use App\Enums\BloodGroup;
use App\Enums\CustomFieldForm;
use App\Enums\Employee\Status;
use App\Enums\Gender;
use App\Enums\MaritalStatus;
use App\Enums\UserStatus;
use App\Http\Resources\Company\BranchResource;
use App\Http\Resources\Company\DepartmentResource;
use App\Http\Resources\Company\DesignationResource;
use App\Http\Resources\CustomFieldResource;
use App\Http\Resources\OptionResource;
use App\Models\Company\Branch;
use App\Models\Company\Department;
use App\Models\Company\Designation;
use App\Models\Contact;
use App\Models\CustomField;
use App\Models\Employee\Employee;
use App\Models\Employee\Record;
use App\Models\Option;
use App\Models\Team\Role;
use App\Models\User;
use App\Support\FormatCodeNumber;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Validation\ValidationException;
use Spatie\Permission\Models\Role as SpatieRole;

class EmployeeService
{
    use EnsureUniqueUserEmail, FormatCodeNumber;

    private function codeNumber()
    {
        $numberPrefix = config('config.employee.code_number_prefix');
        $numberSuffix = config('config.employee.code_number_suffix');
        $digit = config('config.employee.code_number_digit', 0);

        $numberFormat = $numberPrefix.'%NUMBER%'.$numberSuffix;
        $codeNumber = (int) Employee::byTeam()->whereNumberFormat($numberFormat)->max('number') + 1;

        return $this->getCodeNumber(number: $codeNumber, digit: $digit, format: $numberFormat);
    }

    private function validateCodeNumber(Request $request, ?string $uuid = null): array
    {
        $existingCodeNumber = Employee::byTeam()->whereCodeNumber($request->code_number)->when($uuid, function ($q, $uuid) {
            $q->where('uuid', '!=', $uuid);
        })->exists();

        if ($existingCodeNumber) {
            throw ValidationException::withMessages(['code_number' => trans('global.duplicate', ['attribute' => trans('employee.props.code_number')])]);
        }

        $codeNumberDetail = $this->codeNumber();

        return $request->code_number == Arr::get($codeNumberDetail, 'code_number') ? $codeNumberDetail : [
            'code_number' => $request->code_number,
        ];
    }

    public function preRequisite(Request $request): array
    {
        $codeNumber = Arr::get($this->codeNumber(), 'code_number');

        $genders = Gender::getOptions();

        $maritalStatuses = MaritalStatus::getOptions();

        $bloodGroups = BloodGroup::getOptions();

        $statuses = Status::getOptions();

        $employeeTypes = [
            ['label' => trans('global.new', ['attribute' => trans('employee.employee')]), 'value' => 'new'],
            ['label' => trans('global.existing', ['attribute' => trans('employee.employee')]), 'value' => 'existing'],
        ];

        $customFields = CustomFieldResource::collection(CustomField::query()
            ->byTeam()
            ->whereForm(CustomFieldForm::EMPLOYEE)
            ->orderBy('position')
            ->get());

        $roles = Role::selectOption();

        $departments = DepartmentResource::collection(Department::query()
            ->byTeam()
            ->get());

        $designations = DesignationResource::collection(Designation::query()
            ->byTeam()
            ->get());

        $branches = BranchResource::collection(Branch::query()
            ->byTeam()
            ->get());

        $employmentStatuses = OptionResource::collection(Option::query()
            ->byTeam()
            ->whereType('employment_status')
            ->get());

        return compact('codeNumber', 'genders', 'maritalStatuses', 'bloodGroups', 'statuses', 'employeeTypes', 'customFields', 'roles', 'departments', 'designations', 'branches', 'employmentStatuses');
    }

    public function create(Request $request): Employee
    {
        \DB::beginTransaction();

        if ($request->employee_type == 'new') {
            $contact = (new CreateContact)->execute($request->all());

            $request->merge([
                'contact_id' => $contact->id,
            ]);
        }

        $employee = Employee::forceCreate($this->formatParams($request));

        if ($request->role_ids) {
            $user = $employee->contact->user;

            if ($user) {
                $user->assignRole(SpatieRole::find($request->role_ids));
            }
        }

        $employeeRecord = Record::forceCreate([
            'employee_id' => $employee->id,
            'department_id' => $request->department_id,
            'designation_id' => $request->designation_id,
            'branch_id' => $request->branch_id,
            'employment_status_id' => $request->employment_status_id,
            'start_date' => $request->joining_date,
        ]);

        if ($request->employee_type == 'new' && $request->boolean('create_user_account')) {
            $this->ensureEmailDoesntBelongToOtherContact($contact, $request->email);

            $this->ensureEmailDoesntBelongToUserContact($request->email);

            if (! $contact->email) {
                $contact->email = $request->email;
                $contact->save();
            }

            $user = User::forceCreate([
                'name' => $contact->name,
                'email' => $contact->email,
                'username' => $request->username,
                'password' => bcrypt($request->password),
                'status' => UserStatus::ACTIVATED,
            ]);

            $user->assignRole(SpatieRole::find($request->role_ids));

            $contact->user_id = $user->id;
            $contact->save();
        }

        \DB::commit();

        if ($request->employee_type == 'new' && $request->boolean('create_user_account')) {
            $this->sendWelcomeNotification($employee, [
                'name' => $contact->name,
                'employee_code' => $employee->code_number,
                'designation' => $request->designation_name,
                'department' => $request->department_name,
                'branch' => $request->branch_name,
                'employment_status' => $request->employment_status_name,
                'username' => $request->username,
                'password' => $request->password,
            ]);
        }

        return $employee;
    }

    private function sendWelcomeNotification(Employee $employee, array $params = [])
    {
        $template = 'employee-account-created';

        (new SendMailTemplate)->execute(
            email: $employee->contact->email,
            code: $template,
            variables: [
                ...$params,
                'url' => url('/'),
            ]
        );
    }

    private function formatParams(Request $request, ?Employee $employee = null): array
    {
        $codeNumberDetail = $this->validateCodeNumber($request);

        $formatted = [
            'contact_id' => $request->contact_id,
            'joining_date' => $request->joining_date,
            'number_format' => Arr::get($codeNumberDetail, 'number_format'),
            'number' => Arr::get($codeNumberDetail, 'number'),
            'code_number' => $request->code_number,
        ];

        return $formatted;
    }

    public function update(Request $request, Employee $employee): void
    {
        $contact = $employee->contact;

        $existingContact = Contact::byTeam()->where('uuid', '!=', $contact->uuid)
            ->whereFirstName($request->input('first_name', $contact->first_name))
            ->whereMiddleName($request->input('middle_name', $contact->middle_name))
            ->whereThirdName($request->input('third_name', $contact->third_name))
            ->whereLastName($request->input('last_name', $contact->last_name))
            ->whereContactNumber($request->input('contact_number', $contact->contact_number))
            ->count();

        if ($existingContact) {
            throw ValidationException::withMessages(['message' => trans('employee.exists')]);
        }

        \DB::beginTransaction();

        (new UpdateContact)->execute($employee->contact, $request->all());

        \DB::commit();
    }

    public function deletable(Employee $employee): void
    {
        if ($employee->contact?->user?->hasRole('admin') && ! auth()->user()->is_default) {
            throw ValidationException::withMessages(['message' => trans('employee.errors.permission_denied')]);
        }
    }
}
