<?php

namespace App\Services\Saas\PaymentGateway;

use App\Concerns\Saas\HasPaymentGateway;
use App\Contracts\Saas\PaymentGateway;
use App\Models\Saas\Subscription;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use Illuminate\Validation\ValidationException;

class Paystack implements PaymentGateway
{
    use HasPaymentGateway;

    public function getName(): string
    {
        return 'paystack';
    }

    public function isEnabled(): void
    {
        if (! config('saas.landlord.subscription.enable_paystack', false)) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_operation')]);
        }
    }

    public function getMultiplier(Request $request): float
    {
        return 100;
    }

    public function supportedCurrencies(): array
    {
        return ['NGN', 'GHS'];
    }

    public function unsupportedCurrencies(): array
    {
        return [];
    }

    public function initiatePayment(Request $request, Subscription $subscription): array
    {
        $this->validateCurrency($subscription, $this->supportedCurrencies(), $this->unsupportedCurrencies());

        $period = $subscription->start_date->formatted.' - '.$subscription->expiry_date->formatted;

        return [
            'token' => $subscription->uuid,
            'amount' => \Price::from($subscription->amount->value * $this->getMultiplier($request), $request->currency),
            'key' => config('saas.landlord.subscription.paystack_client'),
            'currency' => $request->currency,
            'name' => $request->plan->name.' '.trans('saas.subscription.subscription'),
            'email' => $subscription->tenant->email,
            'description' => $period,
            'icon' => config('saas.icon'),
        ];
    }

    private function getSubscription(Request $request)
    {
        $subscription = Subscription::query()
            ->whereTenantId(config('saas.tenant.id'))
            ->whereUuid($request->token)
            ->whereStatus(0)
            ->first();

        if (! $subscription) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_input')]);
        }

        return $subscription;
    }

    public function confirmPayment(Request $request): Subscription
    {
        $subscription = $this->getSubscription($request);

        $response = Http::withToken(config('saas.landlord.subscription.paystack_secret'))
            ->get('https://api.paystack.co/transaction/verify/'.$request->payment_id);

        $body = json_decode($response->body(), true);

        if (Arr::get($body, 'status') !== true) {
            throw ValidationException::withMessages(['message' => trans('finance.payment_failed')]);
        }

        if (Arr::get($body, 'data.amount') != $subscription->amount->value * $this->getMultiplier($request)) {
            throw ValidationException::withMessages(['message' => trans('finance.amount_mismatch')]);
        }

        $metaData = Arr::get($body, 'data.metadata');
        $customField = Arr::first(Arr::get($metaData, 'custom_fields'));

        if (Arr::get($customField, 'value') !== $subscription->uuid) {
            throw ValidationException::withMessages(['message' => trans('finance.invalid_token')]);
        }

        $subscription->payment_gateway = [
            'name' => $this->getName(),
            'payment_id' => $request->payment_id,
        ];

        return $subscription;
    }

    public function failPayment(Request $request): Subscription
    {
        $subscription = $this->getSubscription($request);

        $failedLogs = $subscription->failed_logs;
        $failedLogs[] = [
            'name' => $this->getName(),
            'error' => $request->error,
        ];
        $subscription->failed_logs = $failedLogs;
        $subscription->save();

        return $subscription;
    }
}
