<?php

namespace App\Services\Saas\PaymentGateway;

use App\Concerns\Saas\HasPaymentGateway;
use App\Contracts\Saas\PaymentGateway;
use App\Helpers\SysHelper;
use App\Models\Saas\Subscription;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class Payzone implements PaymentGateway
{
    use HasPaymentGateway;

    public function getName(): string
    {
        return 'payzone';
    }

    public function isEnabled(): void
    {
        if (! config('saas.landlord.subscription.enable_payzone', false)) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_operation')]);
        }
    }

    public function getMultiplier(Request $request): float
    {
        return 1;
    }

    public function supportedCurrencies(): array
    {
        return ['MAD'];
    }

    public function unsupportedCurrencies(): array
    {
        return [];
    }

    public function initiatePayment(Request $request, Subscription $subscription): array
    {
        $this->validateCurrency($subscription, $this->supportedCurrencies(), $this->unsupportedCurrencies());

        $merchant = config('saas.landlord.subscription.payzone_merchant');
        $secretKey = config('saas.landlord.subscription.payzone_secret_key');
        $notificationKey = config('saas.landlord.subscription.payzone_notification_key');
        $url = config('saas.landlord.subscription.enable_live_payzone_mode') ? 'https://payzone.ma/pwthree/launch' : 'https://payment-sandbox.payzone.ma/pwthree/launch';

        $period = $subscription->start_date->formatted.' - '.$subscription->expiry_date->formatted;

        $chargeId = strtoupper(Str::random(20));

        $payload = [
            'merchantAccount' => $merchant,
            'timestamp' => time(),
            'skin' => 'vps-1-vue',

            'customerId' => 'payplus-paywall-poc',
            'customerCountry' => 'MA',
            'customerLocale' => 'en_US',

            'chargeId' => $chargeId,
            'price' => $subscription->amount->value * $this->getMultiplier($request),
            'currency' => $subscription->currency,
            // Temporarily testing the price and currency
            // 'price' => 10,
            // 'currency' => 'MAD',
            'description' => $period,

            'chargeProperties' => [
                'param1' => 'subscription',
                'param2' => $subscription->uuid,
            ],
            'mode' => 'DEEP_LINK',
            'paymentMethod' => 'CREDIT_CARD',
            'showPaymentProfiles' => 'true',
            'flowCompletionUrl' => (string) url('/app/subscriptions'),
            'successUrl' => (string) url('/app/subscriptions'),
            'callbackUrl' => (string) url('/subscriptions/payment/payzone/response'),
        ];

        $subscription->payment_gateway = [
            'name' => $this->getName(),
            'charge_id' => $chargeId,
        ];
        $subscription->save();

        $jsonPayload = json_encode($payload);
        $signature = hash('sha256', $secretKey.$jsonPayload);

        return [
            'token' => $subscription->uuid,
            'payload' => $jsonPayload,
            'signature' => $signature,
            'pg_url' => $url,
            'amount' => SysHelper::formatAmount($subscription->amount->value * $this->getMultiplier($request), $request->currency),
            'key' => config('saas.landlord.subscription.payzone_merchant'),
            'currency' => $request->currency,
            'name' => $request->plan->name.' '.trans('saas.subscription.subscription'),
            'description' => $period,
            'icon' => config('saas.icon'),
        ];
    }

    private function getSubscription(Request $request)
    {
        $subscription = Subscription::query()
            ->whereTenantId(config('saas.tenant.id'))
            ->whereUuid($request->token)
            ->whereStatus(0)
            ->first();

        if (! $subscription) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_input')]);
        }

        return $subscription;
    }

    public function confirmPayment(Request $request): Subscription
    {
        $subscription = $this->getSubscription($request);

        throw ValidationException::withMessages(['message' => 'test']);

        return $subscription;
    }

    public function failPayment(Request $request): Subscription
    {
        $subscription = $this->getSubscription($request);

        $failedLogs = $subscription->failed_logs;
        $failedLogs[] = [
            'name' => $this->getName(),
            'error' => $request->error,
        ];
        $subscription->failed_logs = $failedLogs;
        $subscription->save();

        return $subscription;
    }
}
