<?php

namespace App\Services\Saas;

use App\Actions\Saas\GetLimit;
use App\Actions\Saas\GetPaymentGateway;
use App\Actions\Saas\Subscription\Calculate;
use App\Actions\Saas\Subscription\CompleteSubscription;
use App\Actions\Saas\Subscription\CreateSubscription;
use App\Actions\Saas\Subscription\SendConfirmation;
use App\Actions\Saas\Subscription\SendFailureNotification;
use App\Actions\Saas\Subscription\UpdateTenant;
use App\Contracts\Saas\PaymentGateway;
use App\Enums\Saas\PlanFrequency;
use App\Http\Resources\Saas\PlanResource;
use App\Http\Resources\Saas\TenantResource;
use App\Models\Saas\Plan;
use App\Models\Saas\Subscription;
use App\Models\Saas\Tenant;
use Illuminate\Http\Request;
use Illuminate\Pipeline\Pipeline;
use Illuminate\Support\Arr;

class SubscriptionService
{
    public function preRequisite(Request $request): array
    {
        $tenant = Tenant::withActiveSubscriptionId()->find(config('saas.tenant.id'));
        $plan = Plan::find($tenant->plan_id);

        $limits = (new GetLimit)->execute(plan: $plan);

        $tenant = TenantResource::make($tenant);
        $plan = PlanResource::make($plan);

        $plans = PlanResource::collection(Plan::where('features->is_active', true)->where('features->is_visible', true)->get());

        $frequencies = PlanFrequency::getOptions();
        $currencies = collect(Arr::getVar('currencies'))->whereIn('name', explode(',', config('saas.landlord.system.currencies')))->values()->all();

        $paymentGateways = (new GetPaymentGateway)->execute();

        return compact('limits', 'tenant', 'plan', 'plans', 'frequencies', 'currencies', 'paymentGateways');
    }

    public function calculate(Request $request): array
    {
        $data = (new Calculate)->execute($request);

        $activationCharge = Arr::get($data, 'activationCharge', 0);
        $amount = Arr::get($data, 'amount', 0);
        $balance = Arr::get($data, 'balance', 0);
        $taxAmount = Arr::get($data, 'taxAmount', 0);
        $taxLabel = Arr::get($data, 'taxLabel');
        $total = Arr::get($data, 'total', 0);

        return [
            'activation_charge' => \Price::from($activationCharge, $request->currency),
            'amount' => \Price::from($amount, $request->currency),
            'balance' => \Price::from($balance, $request->currency),
            'tax_label' => $taxLabel,
            'tax_amount' => \Price::from($taxAmount, $request->currency),
            'total' => \Price::from($total, $request->currency),
            'start_date' => \Cal::date(Arr::get($data, 'startDate')),
            'end_date' => \Cal::date(Arr::get($data, 'endDate')),
        ];
    }

    public function initiate(Request $request, PaymentGateway $paymentGateway): array|Subscription
    {
        $paymentGateway->isEnabled();

        $data = (new Calculate)->execute($request);

        $subscription = (new CreateSubscription)->execute(request: $request, params: $data);

        if ($subscription->amount === 0) {
            return $this->complete($subscription);
        }

        return $paymentGateway->initiatePayment($request, $subscription);
    }

    private function complete(Subscription $subscription): Subscription
    {
        \DB::beginTransaction();

        $subscription = app(Pipeline::class)
            ->send($subscription)
            ->through([
                CompleteSubscription::class,
                UpdateTenant::class,
                SendConfirmation::class,
            ])
            ->thenReturn();

        \DB::commit();

        return $subscription;
    }

    public function create(Request $request, PaymentGateway $paymentGateway): Subscription
    {
        $subscription = $paymentGateway->confirmPayment($request);

        return $this->complete($subscription);
    }

    public function fail(Request $request, PaymentGateway $paymentGateway): void
    {
        $subscription = $paymentGateway->failPayment($request);

        $subscription = app(Pipeline::class)
            ->send($subscription)
            ->through([
                SendFailureNotification::class,
            ])
            ->thenReturn();
    }
}
