<?php

namespace App\Support;

use App\Models\Audience;
use App\Models\Company\Branch;
use App\Models\Company\Department;
use App\Models\Company\Designation;
use App\Models\Contact;
use App\Models\Employee\Employee;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Validation\ValidationException;

trait HasAudience
{
    public function validateInput(array $params = [])
    {
        $employeeAudienceType = Arr::get($params, 'employee_audience_type');

        $employeeAudiences = [];

        if (Arr::get($params, 'is_public')) {
            $employeeAudienceType = null;

            return compact('employeeAudienceType', 'employeeAudiences');
        }

        if ($employeeAudienceType == 'department_wise') {
            $departments = Department::query()
                ->byTeam()
                ->select('id', 'uuid')
                ->get();

            foreach (Arr::get($params, 'employee_audiences', []) as $employeeAudience) {
                $department = $departments->firstWhere('uuid', $employeeAudience);
                if (! $department) {
                    throw ValidationException::withMessages(['employee_audiences' => __('global.could_not_find', ['attribute' => __('company.department.department')])]);
                } else {
                    $employeeAudiences[] = $department->id;
                }
            }
        } elseif ($employeeAudienceType == 'designation_wise') {
            $designations = Designation::query()
                ->byTeam()
                ->select('id', 'uuid')
                ->get();

            foreach (Arr::get($params, 'employee_audiences', []) as $employeeAudience) {
                $designation = $designations->firstWhere('uuid', $employeeAudience);
                if (! $designation) {
                    throw ValidationException::withMessages(['employee_audiences' => __('global.could_not_find', ['attribute' => __('company.designation.designation')])]);
                } else {
                    $employeeAudiences[] = $designation->id;
                }
            }
        } elseif ($employeeAudienceType == 'branch_wise') {
            $branches = Branch::query()
                ->byTeam()
                ->select('id', 'uuid')
                ->get();

            foreach (Arr::get($params, 'employee_audiences', []) as $employeeAudience) {
                $branch = $branches->firstWhere('uuid', $employeeAudience);
                if (! $branch) {
                    throw ValidationException::withMessages(['employee_audiences' => __('global.could_not_find', ['attribute' => __('company.branch.branch')])]);
                } else {
                    $employeeAudiences[] = $branch->id;
                }
            }
        } elseif ($employeeAudienceType == 'employee_wise') {
            //
        }

        return compact('employeeAudienceType', 'employeeAudiences');
    }

    public function storeAudience(Model $model, array $params = [])
    {
        $employeeAudienceType = Arr::get($params, 'employee_audience_type');
        $employeeAudiences = Arr::get($params, 'employee_audiences', []);

        if ($employeeAudienceType == 'department_wise') {
            foreach ($employeeAudiences as $department) {
                Audience::firstOrCreate([
                    'shareable_id' => $model->id,
                    'shareable_type' => $model->getModelName(),
                    'audienceable_id' => $department,
                    'audienceable_type' => 'Department',
                ]);
            }
        } elseif ($employeeAudienceType == 'designation_wise') {
            foreach ($employeeAudiences as $designation) {
                Audience::firstOrCreate([
                    'shareable_id' => $model->id,
                    'shareable_type' => $model->getModelName(),
                    'audienceable_id' => $designation,
                    'audienceable_type' => 'Designation',
                ]);
            }
        } elseif ($employeeAudienceType == 'branch_wise') {
            foreach ($employeeAudiences as $branch) {
                Audience::firstOrCreate([
                    'shareable_id' => $model->id,
                    'shareable_type' => $model->getModelName(),
                    'audienceable_id' => $branch,
                    'audienceable_type' => 'Branch',
                ]);
            }
        } elseif ($employeeAudienceType == 'employee_wise') {
            foreach ($employeeAudiences as $employee) {
                Audience::firstOrCreate([
                    'shareable_id' => $model->id,
                    'shareable_type' => $model->getModelName(),
                    'audienceable_id' => $employee,
                    'audienceable_type' => 'Employee',
                ]);
            }
        }
    }

    public function updateAudience(Model $model, array $params = [])
    {
        if ($model->is_public) {
            $model->audiences()->delete();

            return;
        }

        $employeeAudienceType = Arr::get($model->audience, 'employee_type');

        if ($employeeAudienceType == 'all') {
            $model->audiences()->whereIn('audienceable_type', ['Department', 'Designation'])->delete();
        }

        if ($employeeAudienceType == 'all') {
            return;
        }

        $model->audiences()->delete();

        $this->storeAudience($model, $params);
    }

    public function prepareAudienceForUpdate(Model $model, array $params = [])
    {
        if ($model->is_public && Arr::get($params, 'is_public')) {
            return;
        }
    }

    public function getContacts(array $params = [])
    {
        $employeeAudienceType = Arr::get($params, 'employee_audience_type');
        $employeeAudiences = Arr::get($params, 'employee_audiences', []);

        $employees = collect([]);
        if ($employeeAudienceType == 'all') {
            $employees = Employee::query()
                ->byTeam()
                ->select('employees.id', 'employees.contact_id')
                ->get();
        } elseif ($employeeAudienceType == 'department_wise') {
            $departments = Department::query()
                ->byTeam()
                ->whereIn('id', $employeeAudiences)
                ->get();

            $employees = Employee::query()
                ->byTeam()
                ->select('employees.id', 'employees.contact_id')
                ->leftJoin('employee_records', function ($join) {
                    $join->on('employees.id', '=', 'employee_records.employee_id')
                        ->on('start_date', '=', \DB::raw("(select start_date from employee_records where employees.id = employee_records.employee_id and start_date <= '".today()->toDateString()."' order by start_date desc limit 1)"))
                        ->join('departments', 'employee_records.department_id', '=', 'departments.id');
                })
                ->whereIn('departments.id', $departments->pluck('id')->toArray())
                ->get();
        } elseif ($employeeAudienceType == 'designation_wise') {
            $designations = Designation::query()
                ->byTeam()
                ->whereIn('id', $employeeAudiences)
                ->get();

            $employees = Employee::query()
                ->byTeam()
                ->select('employees.id', 'employees.contact_id')
                ->leftJoin('employee_records', function ($join) {
                    $join->on('employees.id', '=', 'employee_records.employee_id')
                        ->on('start_date', '=', \DB::raw("(select start_date from employee_records where employees.id = employee_records.employee_id and start_date <= '".today()->toDateString()."' order by start_date desc limit 1)"))
                        ->join('designations', 'employee_records.designation_id', '=', 'designations.id');
                })
                ->whereIn('designations.id', $designations->pluck('id')->toArray())
                ->get();
        } elseif ($employeeAudienceType == 'branch_wise') {
            $branches = Branch::query()
                ->byTeam()
                ->whereIn('id', $employeeAudiences)
                ->get();

            $employees = Employee::query()
                ->byTeam()
                ->select('employees.id', 'employees.contact_id')
                ->leftJoin('employee_records', function ($join) {
                    $join->on('employees.id', '=', 'employee_records.employee_id')
                        ->on('start_date', '=', \DB::raw("(select start_date from employee_records where employees.id = employee_records.employee_id and start_date <= '".today()->toDateString()."' order by start_date desc limit 1)"))
                        ->join('branches', 'employee_records.branch_id', '=', 'branches.id');
                })
                ->whereIn('branches.id', $branches->pluck('id')->toArray())
                ->get();
        }

        $contacts = Contact::query()
            ->whereIn('id', $employees->pluck('contact_id'))
            ->get();

        return $contacts;
    }
}
