<?php

use App\Enums\DeviceType;
use App\Helpers\CalHelper;
use App\Helpers\SysHelper;
use App\Models\Attendance\Timesheet;
use App\Models\Attendance\WorkShift;
use App\Models\Device;
use App\Models\Employee\WorkShift as EmployeeWorkShift;
use Database\Seeders\AssignPermissionSeeder;
use Database\Seeders\Company\BranchSeeder;
use Database\Seeders\Company\DepartmentSeeder;
use Database\Seeders\Company\DesignationSeeder;
use Database\Seeders\ConfigSeeder;
use Database\Seeders\DefaultContactSeeder;
use Database\Seeders\Employee\DefaultEmployeeSeeder;
use Database\Seeders\OptionSeeder;
use Database\Seeders\PermissionSeeder;
use Database\Seeders\RoleSeeder;
use Database\Seeders\TeamSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;

use function Pest\Laravel\postJson;
use function Pest\Laravel\seed;

uses(RefreshDatabase::class);

uses()->group('timesheet');

dataset('deviceTimesheetFormData', [
    'default' => [
        [
            'token' => 'test-token',
            'employee_code' => 'ESM002',
        ],
    ],
]);

beforeEach(function () {
    seed([
        TeamSeeder::class,
        RoleSeeder::class,
        PermissionSeeder::class,
        AssignPermissionSeeder::class,
        ConfigSeeder::class,
        OptionSeeder::class,
        DepartmentSeeder::class,
        DesignationSeeder::class,
        BranchSeeder::class,
        DefaultContactSeeder::class,
        DefaultEmployeeSeeder::class,
    ]);

    $workShift = WorkShift::forceCreate([
        'name' => 'Night Shift',
        'code' => 'NS',
        'team_id' => 1,
        'records' => [
            [
                'day' => 'tuesday',
                'is_holiday' => false,
                'is_overnight' => true,
                'start_time' => CalHelper::storeDateTime('22:00:00')->toTimeString(),
                'end_time' => CalHelper::storeDateTime('06:00:00')->toTimeString(),
            ],
            [
                'day' => 'sunday',
                'is_holiday' => true,
                'is_overnight' => false,
                'start_time' => null,
                'end_time' => null,
            ],
        ],
        'description' => 'Night Shift',
    ]);

    EmployeeWorkShift::forceCreate([
        'employee_id' => 2,
        'start_date' => today()->subMonth(1)->startOfMonth()->toDateString(),
        'end_date' => today()->endOfMonth()->toDateString(),
        'work_shift_id' => $workShift->id,
    ]);

    Device::factory()->create([
        'name' => 'Test Device',
        'type' => DeviceType::BIOMETRIC_ATTENDANCE->value,
        'code' => 'test-device',
        'ip_address' => '127.0.0.1',
        'token' => 'test-token',
    ]);

    setConfig([
        'name' => 'attendance',
        'value' => ['allow_employee_clock_in_out_via_device' => true],
    ]);

    SysHelper::setTeam(1);
});

afterEach(function () {});

test('before workshift timesheet can be stored', function (array $data, mixed $dateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 21:30:00']);

test('between workshift timesheet can be stored', function (array $data, mixed $dateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 22:30:00']);

test('after workshift timesheet can be stored', function (array $data, mixed $dateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->addDay(1)->toDateString().' 7:00:00']);

test('after 9 workshift timesheet can be stored for next day', function (array $data, mixed $dateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->addDay(1)->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->addDay(1)->toDateString().' 9:10:00']);

test('duplicate timesheet will be skipped', function (array $data, mixed $dateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $timesheet = Timesheet::first();
    expect(Timesheet::count())->toBe(1);

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 22:00:00']);

test('out at timesheet can be stored', function (array $data, mixed $dateTime, mixed $outDateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $outDateTime],
    ]);

    expect($response->status())->toBe(200);
    expect(Timesheet::count())->toBe(1);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBe(CalHelper::storeDateTime($outDateTime)->toDateTimeString());
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 22:05:00'])->with([getDateForTimesheet()->addDay(1)->toDateString().' 6:05:00']);

test('time should be greater than previous stored in time', function (array $data, mixed $dateTime, mixed $anotherDateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $anotherDateTime],
    ]);

    expect($response->status())->toBe(200);
    expect(Timesheet::count())->toBe(1);
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 22:05:00'])->with([getDateForTimesheet()->toDateString().' 22:00:00']);

test('time should be greater than previous stored out time', function (array $data, mixed $dateTime, mixed $outDateTime, mixed $anotherDateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $outDateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $anotherDateTime],
    ]);

    expect($response->status())->toBe(200);
    expect(Timesheet::count())->toBe(1);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBe(CalHelper::storeDateTime($outDateTime)->toDateTimeString());
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 22:05:00'])->with([getDateForTimesheet()->addDay(1)->toDateString().' 6:00:00'])->with([getDateForTimesheet()->addDay(1)->toDateString().' 5:00:00']);

test('next day timesheet can be stored', function (array $data, mixed $dateTime, mixed $nextDatetime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $nextDatetime],
    ]);

    expect($response->status())->toBe(200);
    expect(Timesheet::count())->toBe(2);

    $timesheet = Timesheet::find(2);

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->addDay(1)->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($nextDatetime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 22:05:00'])->with([getDateForTimesheet()->addDay(1)->toDateString().' 9:01:00']);

function getDateForTimesheet()
{
    return today()->subMonth(1)->lastOfMonth()->next('tuesday');
}

function storeDeviceTimesheet(array $data = [])
{
    return postJson(route('device.timesheet.store'), $data);
}
