<?php

use App\Enums\DeviceType;
use App\Helpers\CalHelper;
use App\Helpers\SysHelper;
use App\Models\Attendance\Timesheet;
use App\Models\Attendance\WorkShift;
use App\Models\Device;
use App\Models\Employee\WorkShift as EmployeeWorkShift;
use Database\Seeders\AssignPermissionSeeder;
use Database\Seeders\Company\BranchSeeder;
use Database\Seeders\Company\DepartmentSeeder;
use Database\Seeders\Company\DesignationSeeder;
use Database\Seeders\ConfigSeeder;
use Database\Seeders\DefaultContactSeeder;
use Database\Seeders\Employee\DefaultEmployeeSeeder;
use Database\Seeders\OptionSeeder;
use Database\Seeders\PermissionSeeder;
use Database\Seeders\RoleSeeder;
use Database\Seeders\TeamSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;

use function Pest\Laravel\postJson;
use function Pest\Laravel\seed;

uses(RefreshDatabase::class);

uses()->group('timesheet');

dataset('deviceTimesheetFormData', [
    'default' => [
        [
            'token' => 'test-token',
            'employee_code' => 'ESM002',
        ],
    ],
]);

beforeEach(function () {
    seed([
        TeamSeeder::class,
        RoleSeeder::class,
        PermissionSeeder::class,
        AssignPermissionSeeder::class,
        ConfigSeeder::class,
        OptionSeeder::class,
        DepartmentSeeder::class,
        DesignationSeeder::class,
        BranchSeeder::class,
        DefaultContactSeeder::class,
        DefaultEmployeeSeeder::class,
    ]);

    $workShift = WorkShift::forceCreate([
        'name' => 'First Shift',
        'code' => 'FS',
        'team_id' => 1,
        'records' => [
            [
                'day' => 'monday',
                'is_holiday' => false,
                'is_overnight' => false,
                'start_time' => CalHelper::storeDateTime('08:00:00')->toTimeString(),
                'end_time' => CalHelper::storeDateTime('16:00:00')->toTimeString(),
            ],
            [
                'day' => 'sunday',
                'is_holiday' => true,
                'is_overnight' => false,
                'start_time' => null,
                'end_time' => null,
            ],
        ],
        'description' => 'First Shift',
    ]);

    EmployeeWorkShift::forceCreate([
        'employee_id' => 2,
        'start_date' => today()->startOfMonth()->toDateString(),
        'end_date' => today()->endOfMonth()->toDateString(),
        'work_shift_id' => $workShift->id,
    ]);

    Device::factory()->create([
        'name' => 'Test Device',
        'type' => DeviceType::BIOMETRIC_ATTENDANCE->value,
        'code' => 'test-device',
        'ip_address' => '127.0.0.1',
        'token' => 'test-token',
    ]);

    setConfig([
        'name' => 'attendance',
        'value' => ['allow_employee_clock_in_out_via_device' => true],
    ]);

    SysHelper::setTeam(1);
});

afterEach(function () {});

test('token is required', function (array $data, mixed $token) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['token' => $token],
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->toHaveKey('token');
})->with('deviceTimesheetFormData')->with(['']);

test('valid token is required', function (array $data, mixed $token) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['token' => $token],
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->code->toBe([101]);
})->with('deviceTimesheetFormData')->with(['test']);

test('employee code is required', function (array $data, mixed $employeeCode) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['employee_code' => $employeeCode],
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->toHaveKey('employee_code');
})->with('deviceTimesheetFormData')->with(['']);

test('valid employee code is required', function (array $data, mixed $employeeCode) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['employee_code' => $employeeCode],
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->code->toBe([102]);
})->with('deviceTimesheetFormData')->with(['test']);

test('validate datetime is optional', function (array $data, mixed $dateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->code->toBe([103]);
})->with('deviceTimesheetFormData')->with(['test']);

test('before workshift timesheet can be stored', function (array $data, mixed $dateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 7:00:00']);

test('between workshift timesheet can be stored', function (array $data, mixed $dateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 9:00:00']);

test('after workshift timesheet can be stored', function (array $data, mixed $dateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 17:00:00']);

test('duplicate timesheet will be skipped', function (array $data, mixed $dateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $timesheet = Timesheet::first();
    expect(Timesheet::count())->toBe(1);

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 17:00:00']);

test('out at timesheet can be stored', function (array $data, mixed $dateTime, mixed $outDateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $outDateTime],
    ]);

    expect($response->status())->toBe(200);
    expect(Timesheet::count())->toBe(1);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBe(CalHelper::storeDateTime($outDateTime)->toDateTimeString());
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 8:05:00'])->with([getDateForTimesheet()->toDateString().' 16:05:00']);

test('time should be greater than previous stored in time', function (array $data, mixed $dateTime, mixed $anotherDateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $anotherDateTime],
    ]);

    expect($response->status())->toBe(200);
    expect(Timesheet::count())->toBe(1);
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 8:05:00'])->with([getDateForTimesheet()->toDateString().' 8:00:00']);

test('time should be greater than previous stored out time', function (array $data, mixed $dateTime, mixed $outDateTime, mixed $anotherDateTime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $outDateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $anotherDateTime],
    ]);

    expect($response->status())->toBe(200);
    expect(Timesheet::count())->toBe(1);

    $timesheet = Timesheet::first();

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($dateTime)->toDateTimeString())
        ->out_at->toBe(CalHelper::storeDateTime($outDateTime)->toDateTimeString());
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 8:05:00'])->with([getDateForTimesheet()->toDateString().' 16:00:00'])->with([getDateForTimesheet()->toDateString().' 15:00:00']);

test('next day timesheet can be stored', function (array $data, mixed $dateTime, mixed $nextDatetime) {
    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $dateTime],
    ]);

    expect($response->status())->toBe(200);

    $response = storeDeviceTimesheet([
        ...$data,
        ...['date_time' => $nextDatetime],
    ]);

    expect($response->status())->toBe(200);
    expect(Timesheet::count())->toBe(2);

    $timesheet = Timesheet::find(2);

    expect($timesheet)
        ->employee_id->toBe(2)
        ->date->toBe(getDateForTimesheet()->addDay(1)->toDateString())
        ->in_at->toBe(CalHelper::storeDateTime($nextDatetime)->toDateTimeString())
        ->out_at->toBeNull();
})->with('deviceTimesheetFormData')->with([getDateForTimesheet()->toDateString().' 8:05:00'])->with([getDateForTimesheet()->addDay(1)->toDateString().' 9:00:00']);

function getDateForTimesheet()
{
    return today()->subMonth(1)->lastOfMonth()->next('monday');
}

function storeDeviceTimesheet(array $data = [])
{
    return postJson(route('device.timesheet.store'), $data);
}
