<?php

use App\Enums\OptionType;
use App\Helpers\SysHelper;
use App\Models\Document;
use App\Models\Employee\Employee;
use App\Models\Option;
use App\Models\User;
use Database\Seeders\AssignPermissionSeeder;
use Database\Seeders\Company\BranchSeeder;
use Database\Seeders\Company\DepartmentSeeder;
use Database\Seeders\Company\DesignationSeeder;
use Database\Seeders\DefaultContactSeeder;
use Database\Seeders\Employee\DefaultEmployeeSeeder;
use Database\Seeders\OptionSeeder;
use Database\Seeders\PermissionSeeder;
use Database\Seeders\RoleSeeder;
use Database\Seeders\TeamSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

use function Pest\Laravel\deleteJson;
use function Pest\Laravel\get;
use function Pest\Laravel\getJson;
use function Pest\Laravel\patchJson;
use function Pest\Laravel\postJson;
use function Pest\Laravel\seed;

uses(RefreshDatabase::class);

uses()->group('employee');

dataset('formData', [
    'default' => [
        [
            'title' => 'First Document Title',
            'start_date' => today()->subYear(1)->startOfYear()->toDateString(),
            'end_date' => today()->subYear(1)->endOfYear()->toDateString(),
            'description' => 'First Document Description',
            'media' => [],
            'media_token' => (string) Str::uuid(),
            'media_hash' => Str::random(10),
        ],
    ],
]);

beforeEach(function () {
    seed([
        TeamSeeder::class,
        RoleSeeder::class,
        PermissionSeeder::class,
        AssignPermissionSeeder::class,
        OptionSeeder::class,
        DepartmentSeeder::class,
        DesignationSeeder::class,
        BranchSeeder::class,
        DefaultContactSeeder::class,
        DefaultEmployeeSeeder::class,
    ]);

    SysHelper::setTeam(1);

    $this->user = loginAsAdmin();
});

afterEach(function () {});

test('valid employee is required for document', function () {
    $response = listDocument(['employee' => 'test']);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->toHaveKey('message');
});

test('user cannot fetch document details if employee not accessible', function () {
    $user = User::whereEmail('staff@staff.com')->first();

    login($user);

    $response = listDocument();

    expect($response->status())->toBe(403);
});

test('user can fetch document pre-requisites', function () {
    $response = documentPreRequisite();

    expect($response->status())->toBe(200);
    expect($response->json())->toHaveKeys([]);
});

test('user cannot list documents without permission', function () {
    $user = createUser();
    login($user);

    $response = listDocument();

    expect($response->status())->toBe(403);
});

test('user cannot list documents without team', function () {
    getJson(route('employees.documents.index', ['employee' => 'test']))->assertStatus(422)->assertJsonStructure(['message']);
});

test('user can list documents', function () {
    $response = listDocument();

    expect($response->status())->toBe(200);
    expect($response->json())->toHaveKeys(getPaginationKeys());
});

test('user can filter documents by title', function (array $data, string $filter) {
    createDocument($data);

    $response = listDocument(['title' => $filter]);

    expect($response->status())->toBe(200);
    expect($response->json('data'))->toHaveCount(1);
})->with('formData')->with(['first']);

test('valid document type is required', function (array $data, mixed $type) {
    $response = createDocument([
        ...$data,
        ...['type' => $type],
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->toHaveKey('type');
})->with('formData')->with(['', 'a', 'test']);

test('valid document title is required', function (array $data, mixed $title) {
    $response = createDocument([
        ...$data,
        ...['title' => $title],
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->toHaveKey('title');
})->with('formData')->with(['', 'a', Str::random(101)]);

test('valid document start date is required', function (array $data, mixed $startDate) {
    $response = createDocument([
        ...$data,
        ...['start_date' => $startDate],
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->toHaveKey('start_date');
})->with('formData')->with(['a', 'test']);

test('valid document end date is required', function (array $data, mixed $endDate) {
    $response = createDocument([
        ...$data,
        ...['end_date' => $endDate],
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->toHaveKey('end_date');
})->with('formData')->with(['a', 'test', today()->subYears(2)->toDateString()]);

test('valid document description is optional', function (array $data, mixed $description) {
    $response = createDocument([
        ...$data,
        ...['description' => $description],
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->toHaveKey('description');
})->with('formData')->with(['a', Str::random(501)]);

test('user can create document', function (array $data) {
    $response = createDocument([
        ...$data,
    ]);

    expect($response->status())->toBe(200);

    $document = $response->json('document');

    expect($document)
        ->uuid->toBeUuid()
        ->title->toEqual($data['title'])
        ->start_date->value->toEqual($data['start_date'])
        ->end_date->value->toEqual($data['end_date'])
        ->description->toEqual($data['description']);
})->with('formData');

test('user cannot create document with duplicate title', function (array $data) {
    $response = createDocument([
        ...$data,
    ]);

    expect($response->status())->toBe(200);

    $response = createDocument([
        ...$data,
    ]);

    expect($response->status())->toBe(422);
    expect($response->json('errors'))->toHaveKeys(['title']);
})->with('formData');

test('user can fetch document', function (array $data) {
    $response = createDocument($data);

    $response = fetchDocument($response->json('document.uuid'));

    expect($response->status())->toBe(200);
    expect($response->json())
        ->uuid->toBeUuid()
        ->title->toEqual($data['title'])
        ->start_date->value->toEqual($data['start_date'])
        ->end_date->value->toEqual($data['end_date'])
        ->description->toEqual($data['description']);
})->with('formData');

test('user can update document', function (array $data, array $updateData) {
    $response = createDocument($data);

    $uuid = $response->json('document.uuid');

    $response = updateDocument($uuid, [
        ...$data,
        ...$updateData,
    ]);

    expect($response->status())->toBe(200);

    $response = fetchDocument($uuid);

    expect($response->json())
        ->uuid->toBeUuid()
        ->title->toEqual($updateData['title'])
        ->start_date->value->toEqual($updateData['start_date'])
        ->end_date->value->toEqual($updateData['end_date'])
        ->description->toEqual($updateData['description']);
})->with('formData')->with([
    [[
        'title' => 'Second Document Title',
        'description' => 'Second Document Description',
        'start_date' => today()->subYears(2)->startOfYear()->toDateString(),
        'end_date' => today()->subYears(2)->endOfYear()->toDateString(),
        'media' => [],
        'media_token' => (string) Str::uuid(),
        'media_hash' => Str::random(10),
    ]],
]);

test('user can delete document', function (array $data) {
    $response = createDocument($data);

    $response = deleteDocument($response->json('document.uuid'));

    expect($response->status())->toBe(200);
    expect(Document::count())->toBe(0);
})->with('formData');

test('user can print document', function () {
    $employee = getEmployeeForDocument();

    $response = get(route('employees.documents.export', ['employee' => $employee->uuid]));

    $response->assertStatus(200);
});

function getEmployeeForDocument()
{
    return Employee::whereHas('contact', function ($q) {
        $q->whereEmail('manager@manager.com');
    })->first();
}

function mergeDocumentFormData($data)
{
    if (! array_key_exists('type', $data)) {
        $data['type'] = Option::query()
            ->whereType(OptionType::DOCUMENT_TYPE->value)
            ->first()
            ->uuid;
    }

    return $data;
}

function documentPreRequisite()
{
    $employee = getEmployeeForDocument();
    $employeeUuid = $employee->uuid;

    return getJson(route('employees.documents.preRequisite', ['employee' => $employeeUuid]));
}

function listDocument(array $data = [])
{
    $employee = getEmployeeForDocument();
    $employeeUuid = Arr::get($data, 'employee', $employee->uuid);

    $data = array_merge($data, ['employee' => $employeeUuid]);

    return getJson(route('employees.documents.index', $data));
}

function createDocument(array $data = [])
{
    $employee = getEmployeeForDocument();
    $employeeUuid = Arr::get($data, 'employee', $employee->uuid);

    $data = mergeDocumentFormData($data);

    return postJson(route('employees.documents.store', ['employee' => $employeeUuid]), $data);
}

function fetchDocument(string $uuid)
{
    $employee = getEmployeeForDocument();
    $employeeUuid = $employee->uuid;

    return getJson(route('employees.documents.show', ['employee' => $employeeUuid, 'document' => $uuid]));
}

function updateDocument(string $uuid, array $data = [])
{
    $employee = getEmployeeForDocument();
    $employeeUuid = Arr::get($data, 'employee', $employee->uuid);

    $data = mergeDocumentFormData($data);

    return patchJson(route('employees.documents.update', ['employee' => $employeeUuid, 'document' => $uuid]), $data);
}

function deleteDocument(string $uuid)
{
    $employee = getEmployeeForDocument();
    $employeeUuid = $employee->uuid;

    return deleteJson(route('employees.documents.destroy', ['employee' => $employeeUuid, 'document' => $uuid]));
}
