<?php

use App\Helpers\CalHelper;
use App\Helpers\SysHelper;
use App\Models\Task\Task;
use Database\Seeders\TeamSeeder;
use Illuminate\Foundation\Testing\RefreshDatabase;

use function Pest\Laravel\seed;

uses(RefreshDatabase::class);

uses()->group('task');

beforeEach(function () {
    seed([
        TeamSeeder::class,
    ]);

    SysHelper::setTeam(1);
});

afterEach(function () {});

test('is completed attribute works', function () {
    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => today()->subDay(1)->toDateString(),
    ]);

    expect($task->is_completed)->toBeFalse();

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => today()->subDay(1)->toDateString(),
        'completed_at' => now()->subHour(1)->toDateTimeString(),
    ]);

    expect($task->is_completed)->toBeTrue();

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => today()->subDay(1)->toDateString(),
        'completed_at' => now()->addHour(1)->toDateTimeString(),
    ]);

    expect($task->is_completed)->toBeFalse();
});

test('due date time attribute works', function () {
    $dueDate = today()->subDay(1)->toDateString();
    $dueTime = '11:00:00';

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'due_time' => CalHelper::storeDateTime($dueDate.' '.$dueTime)->toTimeString(),
    ]);

    expect($task->due_date_time->value)->toEqual($dueTime);

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'due_time' => null,
    ]);

    expect($task->due_date_time)->toBeNull();
});

test('due attribute works', function () {
    $dueDate = today()->subDay(1)->toDateString();
    $dueTime = '11:00:00';

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'due_time' => CalHelper::storeDateTime($dueDate.' '.$dueTime)->toTimeString(),
    ]);

    expect($task->due->value)->toEqual($dueDate.' '.$dueTime);

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'due_time' => null,
    ]);

    expect($task->due->value)->toEqual($dueDate);
});

test('is overdue attribute works', function () {
    $dueDate = today()->subDay(1)->toDateString();
    $dueTime = '11:00:00';

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'due_time' => CalHelper::storeDateTime($dueDate.' '.$dueTime)->toTimeString(),
    ]);

    expect($task->is_overdue)->toBeTrue();

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'due_time' => null,
    ]);

    expect($task->is_overdue)->toBeTrue();

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'due_time' => CalHelper::storeDateTime($dueDate.' '.$dueTime)->toTimeString(),
        'completed_at' => CalHelper::storeDateTime($dueDate.' 10:55:00')->toDateTimeString(),
    ]);

    expect($task->is_overdue)->toBeFalse();

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'due_time' => CalHelper::storeDateTime($dueDate.' '.$dueTime)->toTimeString(),
        'completed_at' => CalHelper::storeDateTime(today()->toDateString().now()->addHour(1)->toTimeString())->toDateTimeString(),
    ]);

    expect($task->is_overdue)->toBeTrue();

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'due_time' => CalHelper::storeDateTime($dueDate.' '.$dueTime)->toTimeString(),
        'completed_at' => CalHelper::storeDateTime(today()->toDateString().now()->subHour(1)->toTimeString())->toDateTimeString(),
    ]);

    expect($task->is_overdue)->toBeFalse();
});

test('overdue days attribute works', function () {
    $dueDate = today()->subDay(1)->toDateString();

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
    ]);

    expect($task->overdue_days)->toBe(1);

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'completed_at' => now()->subDay(1)->toDateTimeString(),
    ]);

    expect($task->overdue_days)->toBe(0);

    $task = Task::factory()->create([
        'start_date' => today()->subDays(2)->toDateString(),
        'due_date' => $dueDate,
        'completed_at' => now()->addDay(1)->toDateTimeString(),
    ]);

    expect($task->overdue_days)->toBe(1);
});
