<?php

namespace App\Actions\Saas;

use App\Enums\Saas\TenantStatus;
use App\Models\Saas\Plan;
use App\Models\Saas\Tenant;
use Closure;
use Illuminate\Support\Arr;

class CheckForTenant
{
    public function handle(Tenant $tenant, Closure $next)
    {
        $host = $_SERVER['HTTP_HOST'];

        $tld = Arr::get(parse_url(config('app.url')), 'host');
        $domainName = preg_replace('/^www\./', '', $host);
        $subdomain = current(explode('.', $domainName));

        $isSubdomain = true;
        if ($domainName != $subdomain.'.'.$tld) {
            $isSubdomain = false;
        }

        $tenant = Tenant::query()
            ->withActivePlanId()
            ->when($isSubdomain, function ($q) use ($subdomain) {
                $q->where('domain', $subdomain);
            })
            ->when(! $isSubdomain, function ($q) use ($domainName) {
                $q->whereNotNull('custom_domain')->where('custom_domain', $domainName);
            })
            ->first();

        if (! $tenant) {
            abort('398', trans('saas.errors.invalid_account'));
        }

        if (! in_array($tenant->status->value, TenantStatus::getKeys())) {
            abort('398', trans('saas.errors.invalid_account_status'));
        }

        if (in_array($tenant->status, [TenantStatus::SUSPENDED, TenantStatus::TERMINATED])) {
            abort('398', trans('saas.errors.account_status', ['attribute' => trans('saas.tenant.statuses.'.$tenant->status->value)]));
        }

        if ($tenant->is_trial && $tenant->getMeta('trial_ends_at') < today()->toDateString()) {
            $tenant->is_trial = 0;
            $tenant->save();
        }

        $isExpired = $tenant->expiry_date->value && $tenant->expiry_date->value < today()->toDateString() ? true : false;

        if (! $tenant->is_trial && ! $tenant->expiry_date->value) {
            $isExpired = true;
        }

        if (! $tenant->is_trial && $tenant->expiry_date->value && $tenant->expiry_date->value < today()->toDateString()) {
            $isExpired = true;
        }

        // Check limit before switching plan carefully
        if ($tenant->active_plan_id && $tenant->active_plan_id != $tenant->plan_id) {
            $tenant->plan_id = $tenant->active_plan_id;
            $tenant->save();
        }

        $plan = Plan::find($tenant->plan_id);

        if ($plan->getFeature('is_free')) {
            $isExpired = false;
        }

        if ($tenant->status == TenantStatus::RUNNING && $isExpired) {
            $tenant->status = TenantStatus::EXPIRED;
            $tenant->save();
        }

        if ($tenant->status == TenantStatus::EXPIRED && ! $isExpired) {
            $tenant->status = TenantStatus::RUNNING;
            $tenant->save();
        }

        config([
            'saas.plan' => $plan,
            'saas.tenant' => [
                'id' => $tenant->id,
                'domain' => $tenant->domain,
                'expiry_date' => $tenant->expiry_date->value,
                'is_trial' => $tenant->is_trial,
                'status' => $tenant->status->value,
            ],
            'saas.status' => ! $tenant->database ? true : false,
        ]);

        if (in_array($tenant->status, [TenantStatus::RUNNING, TenantStatus::EXPIRED])) {
            (new SwitchDatabase)->execute($tenant->database);
        }

        return $next($tenant);
    }
}
