<?php

namespace App\Actions\Saas\Subscription;

use App\Enums\Saas\SubscriptionType;
use App\Helpers\CalHelper;
use App\Helpers\SysHelper;
use App\Models\Saas\Plan;
use App\Models\Saas\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Validation\ValidationException;

class CalculateUpgradeSubscription
{
    public function execute(Request $request, Tenant $tenant, Plan $newPlan, Collection $existingSubscriptions)
    {
        $action = SubscriptionType::UPGRADE->value;

        $currency = $request->currency;
        $frequency = $request->frequency;

        if ($currency != $tenant->getMeta('currency')) {
            throw ValidationException::withMessages(['message' => trans('saas.errors.change_currency')]);
        }

        $currentSubscription = $existingSubscriptions->firstWhere('id', $tenant->subscription_id);

        $nextSubscription = $existingSubscriptions->firstWhere('start_date', '>', $currentSubscription->start_date->value);

        if ($nextSubscription) {
            throw ValidationException::withMessages(['message' => trans('saas.subscription.contact_for_upgrade')]);
        }

        $currentPlanActivationCharge = $currentSubscription?->getMeta('activation_charge') ?? 0;
        $currentPlanTaxAmount = Arr::get($currentSubscription?->tax_details ?? [], 'amount', 0);
        $currentPlanAmount = ($currentSubscription?->amount?->value ?: 0) - $currentPlanActivationCharge - $currentPlanTaxAmount;

        if ($currentPlanAmount < 0) {
            $currentPlanAmount = 0;
        }

        $totalDays = CalHelper::dateDiff($currentSubscription->start_date->value, $currentSubscription->expiry_date->value) - 1;
        $perDayAmount = SysHelper::formatAmount(($currentPlanAmount / $totalDays), $tenant->getMeta('frequency'));
        $daysLeft = CalHelper::dateDiff(today()->toDateString(), $currentSubscription->expiry_date->value);

        $balance = SysHelper::formatAmount(($currentPlanAmount - ($perDayAmount * $daysLeft)), $tenant->getMeta('currency'));

        $amount = $newPlan->getPrice($tenant->getMeta('currency'), $currentSubscription->frequency->value);
        $perDayAmount = SysHelper::formatAmount(($amount / $totalDays), $tenant->getMeta('currency'));
        $amount = SysHelper::formatAmount($perDayAmount * $daysLeft, $tenant->getMeta('currency'));

        $total = SysHelper::formatAmount($amount - $balance, $tenant->getMeta('currency'));

        $tax = (new GetTax)->execute(plan: $newPlan, amount: $amount); // Tax on new amount only
        $taxAmount = Arr::get($tax, 'taxAmount', 0);
        $taxAddition = Arr::get($tax, 'taxAddition', 0);
        $taxLabel = Arr::get($tax, 'taxLabel');

        $total += $taxAddition;

        $startDate = today()->toDateString();
        $endDate = $tenant->expiry_date->value;

        if ($total < 0) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_input')]);
        }

        return compact('action', 'amount', 'startDate', 'endDate', 'balance', 'taxAmount', 'taxLabel', 'total');
    }
}
