<?php

namespace App\Services;

use App\Actions\Attendance\StoreTimesheet;
use App\Helpers\CalHelper;
use App\Models\Attendance\Timesheet;
use App\Models\Device;
use App\Models\Employee\Employee;
use App\Support\QrCodeAttendance;
use Carbon\Carbon;
use chillerlan\QRCode\QRCode;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class AttendanceService
{
    use QrCodeAttendance;

    private function checkQrCodeAttendanceEnabled()
    {
        if (! $this->isQrCodeAttendanceEnabled()) {
            throw ValidationException::withMessages(['message' => trans('attendance.qr_code_attendance_disabled')]);
        }
    }

    public function fetchQrCode(Request $request)
    {
        $qrCodeExpiryDuration = (int) config('config.attendance.qr_code_expiry_duration');

        $this->checkQrCodeAttendanceEnabled();

        if (auth()->user()->is_default) {
            $qrCodeText = 'super-admin';
        } elseif (auth()->user()->hasRole('attendance-assistant')) {
            $qrCodeText = 'attendance-assistant';
        } else {
            $employee = Employee::query()
                ->auth()
                ->firstOrFail();

            $employee = Employee::find($employee->id);

            if (! config('config.attendance.has_dynamic_qr_code')) {
                $qrCodeText = $employee->code_number;
            } else {
                if (! $employee->getMeta('qr_code_attendance')) {
                    $employee->setMeta([
                        'qr_code_attendance' => Str::random(16),
                        'qr_code_expiry_at' => now()->addSeconds($qrCodeExpiryDuration)->toDateTimeString(),
                    ]);
                } else {
                    $qrCodeExpiryAt = $employee->getMeta('qr_code_expiry_at') ?? now()->toDateTimeString();

                    if (Carbon::parse($qrCodeExpiryAt)->isPast()) {
                        $employee->setMeta([
                            'qr_code_attendance' => Str::random(16),
                            'qr_code_expiry_at' => now()->addSeconds($qrCodeExpiryDuration)->toDateTimeString(),
                        ]);
                    }
                }

                $employee->save();

                $qrCodeText = $employee->getMeta('qr_code_attendance');
            }
        }

        $qrCode = (new QRCode)->render(
            $qrCodeText
        );

        return $qrCode;
    }

    public function markAttendance(Request $request)
    {
        // No longer required as we are checking in the next step
        // $this->checkQrCodeAttendanceEnabled();

        $this->markEmployeeAttendance($request);
    }

    public function markAnomymousAttendance(Request $request)
    {
        $token = $request->query('token');

        $device = Device::query()
            ->where('token', $token)
            ->first();

        if (! $device) {
            return response()->json([
                'message' => 'Invalid token',
                'code' => 100,
            ], 422);
        }

        $this->markAttendance($request);
    }

    private function markEmployeeAttendance(Request $request)
    {
        $hasDynamicQrCode = config('config.attendance.has_dynamic_qr_code');

        $employee = Employee::query()
            ->when($hasDynamicQrCode, function ($query) use ($request) {
                $query->where('meta->qr_code_attendance', $request->code);
            }, function ($query) use ($request) {
                $query->where('code_number', $request->code);
            })
            ->firstOrFail();

        if (! config('config.attendance.enable_qr_code_attendance')) {
            throw ValidationException::withMessages(['message' => trans('user.errors.permission_denied')]);
        }

        if (config('config.attendance.has_dynamic_qr_code')) {
            $qrCodeExpiryAt = $employee->getMeta('qr_code_expiry_at') ?? now()->toDateTimeString();

            if (Carbon::parse($qrCodeExpiryAt)->isPast()) {
                throw ValidationException::withMessages(['message' => trans('attendance.qr_code_expired')]);
            }
        }

        $datetime = CalHelper::toDateTime(now()->toDateTimeString());
        $date = Carbon::parse($datetime)->format('Y-m-d');

        $params = [
            'ip' => $request->ip(),
        ];

        $durationBetweenClockRequest = config('config.attendance.duration_between_clock_request', 5);

        $lastTimesheet = Timesheet::query()
            ->whereEmployeeId($employee->id)
            ->where('date', today()->toDateString())
            ->where(function ($q) use ($durationBetweenClockRequest) {
                $q->where('in_at', '>=', now()->subMinutes($durationBetweenClockRequest)->toDateTimeString())
                    ->orWhere('out_at', '>=', now()->subMinutes($durationBetweenClockRequest)->toDateTimeString());
            })
            ->exists();

        if ($lastTimesheet) {
            throw ValidationException::withMessages(['message' => trans('attendance.timesheet.recently_marked')]);
        }

        (new StoreTimesheet)->execute($employee, $datetime, $params);

        if (config('config.attendance.has_dynamic_qr_code')) {
            $qrCodeExpiryDuration = (int) config('config.attendance.qr_code_expiry_duration');

            $employee->setMeta([
                'qr_code_attendance' => Str::random(16),
                'qr_code_expiry_at' => now()->addSeconds($qrCodeExpiryDuration)->toDateTimeString(),
            ]);

            $employee->save();
        }
    }
}
