<?php

namespace App\Services\Dashboard;

use App\Helpers\CalHelper;
use App\Models\Calendar\Holiday;
use App\Models\Employee\Employee;
use App\Models\Task\Task;
use App\Models\Utility\Todo;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class UserCalendarService
{
    public function getData(Request $request): array
    {
        $type = $request->query('type');

        if ($type == 'next' && CalHelper::validateDate($request->query('next_month_date'))) {
            $date = $request->query('next_month_date');
        } elseif ($type == 'previous' && CalHelper::validateDate($request->query('last_month_date'))) {
            $date = $request->query('last_month_date');
        } elseif ($request->query('start_date')) {
            if (! CalHelper::validateDate($request->query('start_date'))) {
                $date = today()->toDateString();
            } else {
                $date = Carbon::parse($request->query('start_date'))->endOfMonth()->toDateString();
            }
        } else {
            if (! CalHelper::validateDate($request->date)) {
                $date = today()->toDateString();
            } else {
                $date = $request->date;
            }
        }

        $nextMonthDate = Carbon::parse($date)->startOfMonth()->addMonth()->toDateString();
        $lastMonthDate = Carbon::parse($date)->startOfMonth()->subDay()->toDateString();

        $startOfMonth = Carbon::parse($date)->startOfMonth();
        $endOfMonth = Carbon::parse($date)->endOfMonth();

        $startDate = $startOfMonth->isMonday() ? $startOfMonth : $startOfMonth->previous(Carbon::MONDAY);
        $endDate = $endOfMonth->isSunday() ? $endOfMonth : $endOfMonth->next(Carbon::SUNDAY);

        $holidays = Holiday::query()
            ->betweenPeriod($startDate->toDateString(), $endDate->toDateString())
            ->get();

        $todos = Todo::query()
            ->where('user_id', auth()->id())
            ->where('due_date', '>=', $startDate->toDateString())
            ->where('due_date', '<=', $endDate->toDateString())
            ->get();

        $employee = Employee::query()
            ->auth()
            ->first();

        $tasks = Task::query()
            ->where(function ($q) use ($employee) {
                $q->whereUserId(auth()->id())
                    ->orWhereHas('members', function ($q) use ($employee) {
                        $q->where('employee_id', $employee?->id);
                    });
            })
            ->where('due_date', '>=', $startDate->toDateString())
            ->where('due_date', '<=', $endDate->toDateString())
            ->get();

        $days = [];
        while ($startDate->lte($endDate)) {
            $isToday = $startDate->isSameDay(today()->toDateString());

            $events = [];

            $currentDate = $startDate->toDateString();

            $holiday = $holidays->where('start_date.value', '<=', $currentDate)->where('end_date.value', '>=', $currentDate)->first();

            if ($holiday) {
                $events[] = [
                    'id' => (string) Str::uuid(),
                    'name' => $holiday->name,
                    'type' => 'holiday',
                    'type_label' => trans('calendar.holiday.holiday'),
                    'time' => '',
                    'datetime' => '',
                    'href' => '#',
                ];
            }

            foreach ($todos->where('due_date.value', $startDate->toDateString()) as $todo) {
                $events[] = [
                    'id' => (string) Str::uuid(),
                    'name' => $todo->title,
                    'type' => 'todo',
                    'type_label' => trans('utility.todo.todo'),
                    'time' => '',
                    'datetime' => '',
                    'href' => '#',
                ];
            }

            foreach ($tasks->where('due_date.value', $startDate->toDateString()) as $task) {
                $events[] = [
                    'id' => (string) Str::uuid(),
                    'name' => $task->title,
                    'type' => 'task',
                    'type_label' => trans('task.task'),
                    'time' => '',
                    'datetime' => '',
                    'href' => '#',
                ];
            }

            $days[] = [
                'date' => $startDate->toDateString(),
                'events' => $events,
                'is_selected' => $isToday,
                'is_current_month' => $startDate->isSameMonth($date),
                'is_today' => $isToday,
            ];

            $startDate->addDay();
        }

        return [
            'days' => $days,
            'month' => Carbon::parse($date)->format('F Y'),
            'last_month_date' => $lastMonthDate,
            'next_month_date' => $nextMonthDate,
        ];
    }
}
