<?php

namespace App\Services\Saas\PaymentGateway;

use App\Concerns\Saas\HasPaymentGateway;
use App\Contracts\Saas\PaymentGateway;
use App\Models\Saas\Subscription;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use Illuminate\Validation\ValidationException;

class Paypal implements PaymentGateway
{
    use HasPaymentGateway;

    public function getName(): string
    {
        return 'paypal';
    }

    public function isEnabled(): void
    {
        if (! config('saas.landlord.subscription.enable_paypal', false)) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_operation')]);
        }
    }

    public function getMultiplier(Request $request): float
    {
        return 1;
    }

    public function supportedCurrencies(): array
    {
        return [];
    }

    public function unsupportedCurrencies(): array
    {
        return [];
    }

    public function initiatePayment(Request $request, Subscription $subscription): array
    {
        $this->validateCurrency($subscription, $this->supportedCurrencies(), $this->unsupportedCurrencies());

        $period = $subscription->start_date->formatted.' - '.$subscription->expiry_date->formatted;

        $subscription->payment_gateway = [
            'name' => $this->getName(),
        ];
        $subscription->save();

        return [
            'token' => $subscription->uuid,
            'amount' => \Price::from($subscription->amount->value * $this->getMultiplier($request), $request->currency),
            'key' => config('saas.landlord.subscription.paypal_client'),
            'currency' => $request->currency,
            'name' => $request->plan->name.' '.trans('saas.subscription.subscription'),
            'description' => $period,
            'icon' => config('saas.icon'),
        ];
    }

    private function getSubscription(Request $request)
    {
        $subscription = Subscription::query()
            ->whereTenantId(config('saas.tenant.id'))
            ->whereUuid($request->token)
            ->whereStatus(0)
            ->first();

        if (! $subscription) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_input')]);
        }

        return $subscription;
    }

    public function confirmPayment(Request $request): Subscription
    {
        $subscription = $this->getSubscription($request);

        $clientId = config('saas.landlord.subscription.paypal_client');
        $secret = config('saas.landlord.subscription.paypal_secret');

        $uri = config('saas.landlord.subscription.enable_live_paypal_mode') ? 'https://api-m.paypal.com' : 'https://api-m.sandbox.paypal.com';

        $response = Http::asForm()->withBasicAuth($clientId, $secret)->post($uri.'/v1/oauth2/token', [
            'grant_type' => 'client_credentials',
        ]);

        $accessToken = Arr::get($response->json(), 'access_token');

        $response = Http::withToken($accessToken)->get($uri.'/v2/checkout/orders/'.$request->input('payment_detail.id'));

        $status = Arr::get($response->json(), 'status');

        if ($status != 'COMPLETED') {
            throw ValidationException::withMessages(['message' => trans('saas.subscription.payment_failed')]);
        }

        // Confirm amount

        $subscription->payment_gateway = [
            'name' => $this->getName(),
            'payment_id' => $request->input('payment_detail.id'),
        ];

        return $subscription;
    }

    public function failPayment(Request $request): Subscription
    {
        $subscription = $this->getSubscription($request);

        $failedLogs = $subscription->failed_logs;
        $failedLogs[] = [
            'name' => $this->getName(),
            'error' => $request->error,
        ];
        $subscription->failed_logs = $failedLogs;
        $subscription->save();

        return $subscription;
    }
}
