<?php

namespace App\Services\Saas\PaymentGateway;

use App\Concerns\Saas\HasPaymentGateway;
use App\Contracts\Saas\PaymentGateway;
use App\Models\Saas\Subscription;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Validation\ValidationException;
use Razorpay\Api\Api as RazorpayApi;

class Razorpay implements PaymentGateway
{
    use HasPaymentGateway;

    public function getName(): string
    {
        return 'razorpay';
    }

    public function isEnabled(): void
    {
        if (! config('saas.landlord.subscription.enable_razorpay', false)) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_operation')]);
        }
    }

    public function getMultiplier(Request $request): float
    {
        return 100;
    }

    public function supportedCurrencies(): array
    {
        return [];
    }

    public function unsupportedCurrencies(): array
    {
        return [];
    }

    public function initiatePayment(Request $request, Subscription $subscription): array
    {
        $this->validateCurrency($subscription, $this->supportedCurrencies(), $this->unsupportedCurrencies());

        $period = $subscription->start_date->formatted.' - '.$subscription->expiry_date->formatted;

        return [
            'token' => $subscription->uuid,
            'amount' => \Price::from($subscription->amount->value * $this->getMultiplier($request), $request->currency),
            'key' => config('saas.landlord.subscription.razorpay_client'),
            'currency' => $request->currency,
            'name' => $request->plan->name.' '.trans('saas.subscription.subscription'),
            'description' => $period,
            'icon' => config('saas.icon'),
        ];
    }

    private function getSubscription(Request $request)
    {
        $subscription = Subscription::query()
            ->whereTenantId(config('saas.tenant.id'))
            ->whereUuid($request->token)
            ->whereStatus(0)
            ->first();

        if (! $subscription) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_input')]);
        }

        return $subscription;
    }

    public function confirmPayment(Request $request): Subscription
    {
        $subscription = $this->getSubscription($request);

        $api = new RazorpayApi(config('saas.landlord.subscription.razorpay_client'), config('saas.landlord.subscription.razorpay_secret'));
        $payment = $api->payment->fetch($request->payment_id);
        $payment = $payment->toArray();

        $status = Arr::get($payment, 'status');

        if ($status != 'authorized' || $request->token != Arr::get($payment, 'notes.token')) {
            throw ValidationException::withMessages(['message' => trans('saas.subscription.payment_failed')]);
        }

        // Confirm amount

        $subscription->payment_gateway = [
            'name' => $this->getName(),
            'payment_id' => $request->payment_id,
        ];

        return $subscription;
    }

    public function failPayment(Request $request): Subscription
    {
        $subscription = $this->getSubscription($request);

        $failedLogs = $subscription->failed_logs;
        $failedLogs[] = [
            'name' => $this->getName(),
            'error' => $request->error,
        ];
        $subscription->failed_logs = $failedLogs;
        $subscription->save();

        return $subscription;
    }
}
