<?php

namespace App\Services\Saas\PaymentGateway;

use App\Concerns\Saas\HasPaymentGateway;
use App\Contracts\Saas\PaymentGateway;
use App\Models\Saas\Subscription;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Validation\ValidationException;
use Stripe\PaymentIntent;
use Stripe\Stripe as StripeGateway;
use Stripe\StripeClient;

class Stripe implements PaymentGateway
{
    use HasPaymentGateway;

    public function getName(): string
    {
        return 'stripe';
    }

    public function isEnabled(): void
    {
        if (! config('saas.landlord.subscription.enable_stripe', false)) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_operation')]);
        }
    }

    public function getMultiplier(Request $request): float
    {
        return 100;
    }

    public function supportedCurrencies(): array
    {
        return [];
    }

    public function unsupportedCurrencies(): array
    {
        return [];
    }

    public function initiatePayment(Request $request, Subscription $subscription): array
    {
        $this->validateCurrency($subscription, $this->supportedCurrencies(), $this->unsupportedCurrencies());

        $period = $subscription->start_date->formatted.' - '.$subscription->expiry_date->formatted;

        StripeGateway::setApiKey(config('saas.landlord.subscription.stripe_secret'));

        try {
            $paymentIntent = PaymentIntent::create([
                'amount' => \Price::from($subscription->amount->value * $this->getMultiplier($request))->value,
                'currency' => $request->currency,
                'automatic_payment_methods' => [
                    'enabled' => true,
                ],
            ]);
        } catch (Exception $e) {
            throw ValidationException::withMessages(['message' => $e->getMessage()]);
        }

        $subscription->payment_gateway = [
            'name' => $this->getName(),
            'payment_intent_id' => $paymentIntent->id,
        ];
        $subscription->save();

        return [
            'token' => $subscription->uuid,
            'client_secret' => $paymentIntent->client_secret,
            'amount' => \Price::from($subscription->amount->value * $this->getMultiplier($request), $request->currency),
            'key' => config('saas.landlord.subscription.stripe_client'),
            'currency' => $request->currency,
            'name' => $request->plan->name.' '.trans('saas.subscription.subscription'),
            'description' => $period,
            'icon' => config('saas.icon'),
        ];
    }

    private function getSubscription(Request $request)
    {
        $subscription = Subscription::query()
            ->whereTenantId(config('saas.tenant.id'))
            ->whereUuid($request->token)
            ->whereStatus(0)
            ->first();

        if (! $subscription) {
            throw ValidationException::withMessages(['message' => trans('general.errors.invalid_input')]);
        }

        return $subscription;
    }

    public function confirmPayment(Request $request): Subscription
    {
        $subscription = $this->getSubscription($request);

        $stripe = new StripeClient(config('saas.landlord.subscription.stripe_secret'));

        $payment = $stripe->paymentIntents->retrieve(Arr::get($subscription->payment_gateway, 'payment_intent_id'), []);

        if ($payment->status != 'succeeded') {
            throw ValidationException::withMessages(['message' => trans('saas.subscription.payment_failed')]);
        }

        return $subscription;
    }

    public function failPayment(Request $request): Subscription
    {
        $subscription = $this->getSubscription($request);

        $failedLogs = $subscription->failed_logs;
        $failedLogs[] = [
            'name' => $this->getName(),
            'error' => $request->error,
        ];
        $subscription->failed_logs = $failedLogs;
        $subscription->save();

        return $subscription;
    }
}
